<?php
// Simple Entry Point for Machinery Marketplace

// Autoloading
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../app/Core/Auth.php';
\Core\Auth::start();

spl_autoload_register(function ($class) {
    $path = __DIR__ . '/../app/' . str_replace('\\', '/', $class) . '.php';
    if (file_exists($path)) {
        require_once $path;
    }
});

// Basic Router
$route = $_GET['route'] ?? 'home';

// Basic Layout Helper
function render($view, $data = []) {
    extract($data);
    require_once __DIR__ . '/../views/layouts/main.php';
}

switch ($route) {
    case 'home':
        render('home', ['title' => 'Machinery Marketplace - Industrial Solutions']);
        break;

        
    case 'buy':
    case 'rent':
    case 'sell':
        $type = $route;
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $items_per_page = 15;
        $filters = ['type' => $type];
        
        $total_items = \Models\Listing::getTotalCount($filters);
        $total_pages = ceil($total_items / $items_per_page);
        
        if ($page < 1) $page = 1;
        if ($page > $total_pages && $total_pages > 0) $page = $total_pages;
        
        $offset = ($page - 1) * $items_per_page;
        
        render('listings', [
            'title' => ucfirst($type) . ' Machinery - Machinery Marketplace',
            'type' => $type,
            'listings' => \Models\Listing::getPaginated($filters, $items_per_page, $offset),
            'category_map' => \Models\ProductType::getAllMapped(),
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $total_pages,
                'total_items' => $total_items,
                'items_per_page' => $items_per_page
            ]
        ]);
        break;


    case 'login':
        render('login', ['title' => 'Login - Machinery Marketplace']);
        break;
    case 'login_process':
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $email = $_POST['email'] ?? '';
            $password = $_POST['password'] ?? '';
            $user = \Models\User::findByEmail($email);
            
            if ($user && password_verify($password, $user['password'])) {
                \Core\Auth::login($user);
                \Models\Activity::log('User Login', "{$user['name']} logged into the system", $user['id'], '🔑');
                // Redirect based on role
                if ($user['role'] === 'super_admin') {
                    header('Location: ?route=super_admin');
                } elseif ($user['role'] === 'company_admin') {
                    header('Location: ?route=company_view&id=' . $user['company_id']);
                } elseif ($user['role'] === 'company_operator') {
                    header('Location: ?route=listing_list');
                } else {
                    header('Location: ?route=home');
                }
                exit;
            } else {
                render('login', ['title' => 'Login - Machinery Marketplace', 'error' => 'Invalid email or password']);
            }
        }
        break;
    case 'super_admin':
        \Core\Auth::check('super_admin');
        render('super_admin', [
            'title' => 'Super Admin - Machinery Marketplace',
            'stats' => [
                'companies' => \Models\Company::count(),
                'users' => \Models\User::count('active'),
                'listings' => \Models\Listing::count()
            ]
        ]);
        break;
    case 'company_admin':
        \Core\Auth::check('company_admin');
        render('company_admin', [
            'title' => 'Dashboard - Machinery Marketplace',
            'company' => \Models\Company::find($_SESSION['company_id']),
            'userCount' => count(\Models\User::getAll($_SESSION['company_id'], 'active')),
            'listings' => \Models\Listing::getStats($_SESSION['company_id'])
        ]);
        break;
    // Company CRUD
    case 'company_list':
        \Core\Auth::check('super_admin');
        render('companies/index', [
            'title' => 'Companies - Machinery Marketplace',
            'companies' => \Models\Company::getAll()
        ]);
        break;
    case 'company_view':
        \Core\Auth::check();
        $companyId = $_GET['id'] ?? 0;
        
        // Permission Check: super_admin can view any, company_admin can view only their own
        if ($_SESSION['role'] !== 'super_admin') {
            if ($companyId != $_SESSION['company_id']) {
                die('Access Denied: Insufficient Permissions');
            }
        }
        
        $company = \Models\Company::find($companyId);
        if (!$company) die('Company not found');
        
        // Pagination Logic
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $items_per_page = 10;
        $offset = ($page - 1) * $items_per_page;
        
        $total_items = \Models\Listing::getTotalCount($company['uuid']);
        $total_pages = ceil($total_items / $items_per_page);
        
        // Ensure page is valid
        if ($page < 1) $page = 1;
        if ($page > $total_pages && $total_pages > 0) $page = $total_pages;
        
        $offset = ($page - 1) * $items_per_page;
        
        render('companies/view', [
            'title' => 'Company Report - ' . $company['name'],
            'company' => $company,
            'users' => \Models\User::getAll($companyId),
            'stats' => \Models\Listing::getStats($company['uuid']),
            'listings' => \Models\Listing::getPaginated($company['uuid'], $items_per_page, $offset),
            'category_map' => \Models\ProductType::getAllMapped(),
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $total_pages,
                'total_items' => $total_items,
                'items_per_page' => $items_per_page
            ]
        ]);
        break;

    case 'company_create':
        \Core\Auth::check('super_admin');
        render('companies/form', [
            'title' => 'Register Company - Machinery Marketplace',
            'types' => \Models\CompanyType::getAll(),
            'product_categories' => \Models\ProductType::getAll(),
            'membership_types' => \Models\MembershipType::getAll()
        ]);
        break;
    case 'company_edit':
        \Core\Auth::check();
        $id = $_GET['id'] ?? 0;
        
        // Permission Check
        if ($_SESSION['role'] !== 'super_admin') {
            if ($id != $_SESSION['company_id']) die('Access Denied');
        }
        
        $company = \Models\Company::find($id);
        if (!$company) die('Company not found');
        
        // Get assigned product categories
        $assigned = \Models\Company::getProductTypes($id);
        $assignedIds = array_column($assigned, 'id');
        
        render('companies/form', [
            'title' => 'Edit Company - Machinery Marketplace',
            'company' => $company,
            'types' => \Models\CompanyType::getAll(),
            'product_categories' => \Models\ProductType::getAll(),
            'assigned_categories' => $assignedIds,
            'membership_types' => \Models\MembershipType::getAll()
        ]);
        break;
    case 'company_save':
        \Core\Auth::check();
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            
            // Permission and Status Check
            if ($_SESSION['role'] !== 'super_admin') {
                if ($id != $_SESSION['company_id']) die('Access Denied');
                
                // Fetch current status to ensure it's active
                $currentCompany = \Models\Company::find($id);
                if (!$currentCompany || $currentCompany['status'] !== 'active') {
                    die('Error: Profile updates are only allowed for Active companies.');
                }
                
                // For safety, force the status and type from the database to prevent manual override
                $_POST['status'] = $currentCompany['status'];
                $_POST['type'] = $currentCompany['type'];
            }

            // Handle Logo Upload
            $logoName = $_POST['existing_logo'] ?? null;
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = __DIR__ . '/uploads/logos/';
                $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $logoName = uniqid('logo_') . '.' . $extension;
                move_uploaded_file($_FILES['logo']['tmp_name'], $uploadDir . $logoName);
            }
            $_POST['logo'] = $logoName;
            
            // Handle Banner Upload
            $bannerName = $_POST['existing_banner'] ?? null;
            if (isset($_FILES['banner']) && $_FILES['banner']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = __DIR__ . '/uploads/banners/';
                if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);
                $extension = pathinfo($_FILES['banner']['name'], PATHINFO_EXTENSION);
                $bannerName = uniqid('banner_') . '.' . $extension;
                move_uploaded_file($_FILES['banner']['tmp_name'], $uploadDir . $bannerName);
            }
            $_POST['banner'] = $bannerName;

            if ($id) {
                \Models\Company::update($id, $_POST);
                if (isset($_POST['product_categories'])) {
                    \Models\Company::syncProductTypes($id, $_POST['product_categories']);
                }
                \Models\Activity::log('Company Updated', "Company {$_POST['name']} profile was updated", null, '🏢');
            } else {
                \Core\Auth::check('super_admin'); // Only super admin can create new ones
                \Models\Company::create($_POST);
                $newId = \Core\Database::getInstance()->lastInsertId();
                if (isset($_POST['product_categories'])) {
                    \Models\Company::syncProductTypes($newId, $_POST['product_categories']);
                }
                \Models\Activity::log('New Company', "Registered {$_POST['name']} on the platform", null, '🆕');
            }
            
            if ($_SESSION['role'] === 'super_admin') {
                header('Location: ?route=company_list');
            } else {
                header('Location: ?route=company_admin');
            }
            exit;
        }
        break;

    // Company Type CRUD
    case 'company_type_list':
        \Core\Auth::check('super_admin');
        render('company_types/index', [
            'title' => 'Company Types - Machinery Marketplace',
            'types' => \Models\CompanyType::getAll()
        ]);
        break;
    case 'company_type_create':
        \Core\Auth::check('super_admin');
        render('company_types/form', ['title' => 'Add Company Type - Machinery Marketplace']);
        break;
    case 'company_type_edit':
        \Core\Auth::check('super_admin');
        render('company_types/form', [
            'title' => 'Edit Company Type - Machinery Marketplace',
            'type' => \Models\CompanyType::find($_GET['id'] ?? 0)
        ]);
        break;
    case 'company_type_save':
        \Core\Auth::check('super_admin');
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['id'])) {
                \Models\CompanyType::update($_POST['id'], $_POST['name']);
            } else {
                \Models\CompanyType::create($_POST['name']);
            }
            header('Location: ?route=company_type_list');
            exit;
        }
        break;
    case 'company_type_delete':
        \Core\Auth::check('super_admin');
        \Models\CompanyType::delete($_GET['id'] ?? 0);
        header('Location: ?route=company_type_list');
        exit;
        break;

    // Membership Type CRUD
    case 'membership_type_list':
        \Core\Auth::check('super_admin');
        render('membership_types/index', [
            'title' => 'Membership Types - Machinery Marketplace',
            'types' => \Models\MembershipType::getAll()
        ]);
        break;
    case 'membership_type_create':
        \Core\Auth::check('super_admin');
        render('membership_types/form', [
            'title' => 'Add Membership Type - Machinery Marketplace'
        ]);
        break;
    case 'membership_type_edit':
        \Core\Auth::check('super_admin');
        render('membership_types/form', [
            'title' => 'Edit Membership Type - Machinery Marketplace',
            'type' => \Models\MembershipType::find($_GET['id'] ?? 0)
        ]);
        break;
    case 'membership_type_save':
        \Core\Auth::check('super_admin');
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['id'])) {
                \Models\MembershipType::update($_POST['id'], $_POST);
            } else {
                \Models\MembershipType::create($_POST);
            }
            header('Location: ?route=membership_type_list');
            exit;
        }
        break;
    case 'membership_type_delete':
        \Core\Auth::check('super_admin');
        $result = \Models\MembershipType::delete($_GET['id'] ?? 0);
        if (!$result) {
            die('Error: Cannot delete membership type that is assigned to companies.');
        }
        header('Location: ?route=membership_type_list');
        exit;
        break;

    // Listing CRUD
    case 'listing_view':
        $id = $_GET['id'] ?? 0;
        $listing = \Models\Listing::find($id);
        if (!$listing) die('Product not found');
        
        $images = \Models\ProductImage::getByProduct($id);
        $company = \Models\Company::find($listing['company_id']);
        
        render('listings/view', [
            'title' => $listing['title'] . ' - Machinery Marketplace',
            'listing' => $listing,
            'images' => $images,
            'company' => $company
        ]);
        break;

    case 'listing_list':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        
        // Company filtering
        if ($_SESSION['role'] === 'super_admin') {
            $companyUuid = $_GET['company_id'] ?? null; // Allow filtering by company
        } else {
            $companyUuid = $_SESSION['company_uuid']; // Force their own company
        }
        
        // Pagination Logic
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $items_per_page = 10;
        $offset = ($page - 1) * $items_per_page;
        
        $total_items = \Models\Listing::getTotalCount($companyUuid);
        $total_pages = ceil($total_items / $items_per_page);
        
        // Ensure page is valid
        if ($page < 1) $page = 1;
        if ($page > $total_pages && $total_pages > 0) $page = $total_pages;
        
        // Recalculate offset if page changed
        $offset = ($page - 1) * $items_per_page;

        $listings = \Models\Listing::getPaginated($companyUuid, $items_per_page, $offset);
        $categoryMapper = \Models\ProductType::getAllMapped();
        
        render('listings/index', [
            'title' => 'Machinery Inventory - Machinery Marketplace',
            'listings' => $listings,
            'category_map' => $categoryMapper,
            'companies' => ($_SESSION['role'] === 'super_admin') ? \Models\Company::getAll() : [],
            'selected_company' => $companyUuid,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $total_pages,
                'total_items' => $total_items,
                'items_per_page' => $items_per_page
            ]
        ]);
        break;
    case 'listing_create':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        render('listings/form', [
            'title' => 'Add New Machinery - Machinery Marketplace',
            'categories' => \Models\ProductType::getTree(),
            'brands' => \Models\Brand::getAll(),
            'companies' => ($_SESSION['role'] === 'super_admin') ? \Models\Company::getAll() : [],
            'locations' => [] 
        ]);
        break;
    case 'listing_edit':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        $id = $_GET['id'] ?? '';
        $listing = \Models\Listing::find($id);
        if (!$listing) die('Product not found');
        
        // Restriction for company users
        if ($_SESSION['role'] !== 'super_admin' && $listing['company_id'] !== $_SESSION['company_uuid']) {
            die('Access Denied: You do not have permission to edit this product.');
        }
        
        $assignedBrands = \Models\Listing::getBrands($id);
        $assignedBrandIds = array_column($assignedBrands, 'uuid');

        render('listings/form', [
            'title' => 'Edit Product - Machinery Marketplace',
            'listing' => $listing,
            'categories' => \Models\ProductType::getTree(),
            'brands' => \Models\Brand::getAll(),
            'assigned_brands' => $assignedBrandIds,
            'images' => \Models\ProductImage::getByProduct($id),
            'companies' => ($_SESSION['role'] === 'super_admin') ? \Models\Company::getAll() : [],
            'locations' => [] 
        ]);
        break;
    case 'listing_save':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = $_POST;
            
            // Access control for ownership
            if ($_SESSION['role'] === 'super_admin') {
                if (empty($data['company_id'])) die('Company assignment is required for Super Admin.');
            } else {
                $data['company_id'] = $_SESSION['company_uuid'];
            }

            // Validate required category
            if (empty($data['product_type_id'])) {
                die('Error: Product category is required.');
            }

            // Map UI checkbox to status
            $data['status'] = !empty($data['is_active']) ? 'active' : 'draft';
            
            // Handle Specification Document Upload
            if (isset($_FILES['specification_doc']) && $_FILES['specification_doc']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = __DIR__ . '/uploads/specifications/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0777, true);
                }
                $extension = pathinfo($_FILES['specification_doc']['name'], PATHINFO_EXTENSION);
                $allowedExtensions = ['pdf', 'doc', 'docx'];
                if (in_array(strtolower($extension), $allowedExtensions)) {
                    $specFileName = uniqid('spec_') . '.' . $extension;
                    move_uploaded_file($_FILES['specification_doc']['tmp_name'], $uploadDir . $specFileName);
                    $data['specification_doc'] = $specFileName;
                }
            } elseif (!empty($data['existing_spec_doc'])) {
                // Keep existing file if no new file uploaded
                $data['specification_doc'] = $data['existing_spec_doc'];
            }
            
            if (!empty($data['id'])) {
                // Verify ownership on update if not super admin
                $existing = \Models\Listing::find($data['id']);
                if ($_SESSION['role'] !== 'super_admin' && $existing['company_id'] !== $_SESSION['company_uuid']) {
                    die('Unauthorized update attempt');
                }
                
                $data['updated_by'] = $_SESSION['user_uuid'];
                $result = \Models\Listing::update($data['id'], $data);
            } else {
                $data['created_by'] = $_SESSION['user_uuid'];
                $result = \Models\Listing::create($data);
                $productId = $result;
            }
            
            // If update, ID is in data. If create, result is ID. 
            if (empty($productId) && !empty($data['id'])) {
                $productId = $data['id'];
            }
            
            // Handle Product Images Upload (Gallery)
            if ($productId && !empty($_FILES['images']['name'][0])) {
                $uploadDir = __DIR__ . '/uploads/products/';
                if (!is_dir($uploadDir)) mkdir($uploadDir, 0777, true);
                
                foreach ($_FILES['images']['name'] as $key => $name) {
                    if ($_FILES['images']['error'][$key] === UPLOAD_ERR_OK) {
                        $extension = pathinfo($name, PATHINFO_EXTENSION);
                        if (in_array(strtolower($extension), ['jpg', 'jpeg', 'png', 'webp'])) {
                            $filename = uniqid('prod_') . '.' . $extension;
                            move_uploaded_file($_FILES['images']['tmp_name'][$key], $uploadDir . $filename);
                            
                            // Determine if this should be primary (if no images exist yet)
                            $existingImages = \Models\ProductImage::getByProduct($productId);
                            $isPrimary = empty($existingImages);
                            
                            \Models\ProductImage::create($productId, $filename, $isPrimary);
                        }
                    }
                }
            }
            
            header('Location: ?route=listing_list');
            exit;
        }
        break;
    case 'listing_delete':
        \Core\Auth::check(['company_admin', 'super_admin']);
        $id = $_GET['id'] ?? '';
        $existing = \Models\Listing::find($id);
        if (!$existing) die('Product not found');
        
        if ($_SESSION['role'] !== 'super_admin' && $existing['company_id'] !== $_SESSION['company_uuid']) {
            die('Unauthorized delete attempt');
        }

        \Models\Listing::delete($id);
        header('Location: ?route=listing_list');
        exit;
        break;

    // Brand CRUD (Super Admin)
    case 'brand_list':
        \Core\Auth::check('super_admin');
        render('brands/index', [
            'title' => 'Manage Brands - Machinery Marketplace',
            'brands' => \Models\Brand::getAll()
        ]);
        break;
    case 'brand_create':
        \Core\Auth::check('super_admin');
        render('brands/form', [
            'title' => 'Register Brand - Machinery Marketplace'
        ]);
        break;
    case 'brand_edit':
        \Core\Auth::check('super_admin');
        render('brands/form', [
            'title' => 'Edit Brand - Machinery Marketplace',
            'brand' => \Models\Brand::find($_GET['id'] ?? 0)
        ]);
        break;
    case 'brand_save':
        \Core\Auth::check('super_admin');
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = $_POST;
            
            // Handle Logo Upload
            if (!empty($_FILES['logo']['name'])) {
                $dir = 'uploads/brands/';
                if (!is_dir($dir)) mkdir($dir, 0777, true);
                $ext = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                $filename = time() . '_' . uniqid() . '.' . $ext;
                move_uploaded_file($_FILES['logo']['tmp_name'], $dir . $filename);
                $data['logo'] = $filename;
            } else if (isset($data['id'])) {
                $existing = \Models\Brand::find($data['id']);
                $data['logo'] = $existing['logo'];
            }

            if (!empty($data['id'])) {
                \Models\Brand::update($data['id'], $data);
            } else {
                \Models\Brand::create($data);
            }
            header('Location: ?route=brand_list');
            exit;
        }
        break;
    case 'brand_delete':
        \Core\Auth::check('super_admin');
        \Models\Brand::delete($_GET['id'] ?? 0);
        header('Location: ?route=brand_list');
        exit;
        break;

    // Product Type CRUD

    // Product Type CRUD
    case 'listing_image_primary':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        $imageId = $_GET['id'] ?? '';
        $image = \Models\ProductImage::find($imageId);
        
        if ($image) {
            $product = \Models\Listing::find($image['product_id']);
            // Permission check
            if ($_SESSION['role'] !== 'super_admin' && $product['company_id'] !== $_SESSION['company_uuid']) {
                die('Access Denied');
            }
            \Models\ProductImage::setPrimary($imageId, $image['product_id']);
            header('Location: ?route=listing_edit&id=' . $image['product_id']);
            exit;
        }
        break;

    case 'listing_image_delete':
        \Core\Auth::check(['company_admin', 'company_operator', 'super_admin']);
        $imageId = $_GET['id'] ?? '';
        $image = \Models\ProductImage::find($imageId);
        
        if ($image) {
            $product = \Models\Listing::find($image['product_id']);
            // Permission check
            if ($_SESSION['role'] !== 'super_admin' && $product['company_id'] !== $_SESSION['company_uuid']) {
                die('Access Denied');
            }
            
            \Models\ProductImage::delete($imageId);
            $filePath = __DIR__ . '/uploads/products/' . $image['image_path'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            header('Location: ?route=listing_edit&id=' . $image['product_id']);
            exit;
        }
        break;

    case 'product_type_list':
        \Core\Auth::check('super_admin');
        render('product_types/index', [
            'title' => 'Product Categories - Machinery Marketplace',
            'types' => \Models\ProductType::getTree()
        ]);
        break;
    case 'product_type_create':
        \Core\Auth::check('super_admin');
        render('product_types/form', [
            'title' => 'Add Product Category - Machinery Marketplace',
            'all_categories' => \Models\ProductType::getTree()
        ]);
        break;
    case 'product_type_edit':
        \Core\Auth::check('super_admin');
        render('product_types/form', [
            'title' => 'Edit Product Category - Machinery Marketplace',
            'type' => \Models\ProductType::find($_GET['id'] ?? 0),
            'all_categories' => \Models\ProductType::getTree()
        ]);
        break;
    case 'product_type_save':
        \Core\Auth::check('super_admin');
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['id'])) {
                \Models\ProductType::update($_POST['id'], $_POST);
            } else {
                \Models\ProductType::create($_POST);
            }
            header('Location: ?route=product_type_list');
            exit;
        }
        break;
    case 'product_type_delete':
        \Core\Auth::check('super_admin');
        \Models\ProductType::delete($_GET['id'] ?? 0);
        header('Location: ?route=product_type_list');
        exit;
        break;

    // User CRUD
    case 'user_list':
        \Core\Auth::check();
        $filterCompanyId = $_GET['company_id'] ?? null;
        $status = $_GET['status'] ?? 'active';
        
        // If not super_admin, force filter to their own company
        if ($_SESSION['role'] !== 'super_admin') {
            $filterCompanyId = $_SESSION['company_id'];
        }
        
        render('users/index', [
            'title' => 'Users - Machinery Marketplace',
            'users' => \Models\User::getAll($filterCompanyId, $status),
            'companies' => \Models\Company::getAll(),
            'currentFilter' => $filterCompanyId,
            'currentStatus' => $status
        ]);
        break;
    case 'user_create':
        \Core\Auth::check();
        render('users/form', [
            'title' => 'Create User - Machinery Marketplace',
            'companies' => \Models\Company::getAll(),
            'prefilled_company_id' => $_GET['company_id'] ?? null
        ]);
        break;
    case 'user_edit':
        \Core\Auth::check();
        render('users/form', [
            'title' => 'Edit User - Machinery Marketplace',
            'user' => \Models\User::find($_GET['id'] ?? 0),
            'companies' => \Models\Company::getAll()
        ]);
        break;
    case 'user_save':
        \Core\Auth::check();
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Security: Role Restrictions
            if ($_SESSION['role'] === 'company_admin') {
                $_POST['company_id'] = $_SESSION['company_id'];
                $_POST['role'] = 'company_operator'; // Company admins can only create operators
            }
            
            if (isset($_POST['id'])) {
                // When updating, preserve the original company assignment
                $existingUser = \Models\User::find($_POST['id']);
                if ($existingUser) {
                    $_POST['company_id'] = $existingUser['company_id']; // Force original company
                }
                \Models\User::update($_POST['id'], $_POST);
                \Models\Activity::log('User Updated', "User {$_POST['name']} profile updated", null, '👤');
            } else {
                \Models\User::create($_POST);
                \Models\Activity::log('New User', "Account created for {$_POST['name']}", null, '👤');
            }
            $redirectUrl = '?route=user_list';
            if (!empty($_POST['company_id'])) {
                $redirectUrl .= '&company_id=' . $_POST['company_id'];
            }
            header('Location: ' . $redirectUrl);
            exit;
        }
        break;
    case 'user_suspend':
        \Core\Auth::check('super_admin');
        $id = $_GET['id'] ?? 0;
        $status = $_GET['status'] ?? 'suspended';
        \Models\User::updateStatus($id, $status);
        
        $user = \Models\User::find($id);
        $redirectUrl = '?route=user_list&status=' . $status;
        if ($user && !empty($user['company_id'])) {
            $redirectUrl .= '&company_id=' . $user['company_id'];
        }
        
        \Models\Activity::log('User Status Changed', "User {$user['name']} status set to {$status}", null, '🔄');
        header('Location: ' . $redirectUrl);
        exit;
        break;

    case 'user_delete':
        \Core\Auth::check('super_admin');
        $user = \Models\User::find($_GET['id'] ?? 0);
        $redirectUrl = '?route=user_list';
        if ($user && !empty($user['company_id'])) {
            $redirectUrl .= '&company_id=' . $user['company_id'];
        }
        
        \Models\User::delete($_GET['id'] ?? 0);
        header('Location: ' . $redirectUrl);
        exit;
        break;

    case 'logout':
        \Core\Auth::logout();
        break;

    default:
        http_response_code(404);
        echo "404 Not Found";
        break;
}
