<?php
require_once __DIR__ . "/config/config.php";
require_once __DIR__ . "/app/Core/Database.php";

use Core\Database;

try {
    $db = Database::getInstance();
    $db->beginTransaction();

    echo "Fetching old listings...\n";
    $stmt = $db->query("
        SELECT l.*, c.uuid as company_uuid, pt.uuid as pt_uuid, u.uuid as user_uuid
        FROM listings l
        JOIN companies c ON l.company_id = c.id
        LEFT JOIN product_types pt ON l.product_type_id = pt.id
        LEFT JOIN users u ON u.company_id = c.id AND u.role = 'company_admin' -- Guessing a creator
        LIMIT 1000
    ");
    $listings = $stmt->fetchAll();

    $insertStmt = $db->prepare("
        INSERT INTO products (
            id, company_id, product_type_id, title, slug, description, 
            price, status, listing_type, created_by, created_at
        ) VALUES (
            ?, ?, ?, ?, ?, ?, 
            ?, ?, ?, ?, ?
        )
    ");

    function slugify($text) {
        $text = preg_replace('~[^\pL\d]+~u', '-', $text);
        $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
        $text = preg_replace('~[^-\w]+~', '', $text);
        $text = trim($text, '-');
        $text = preg_replace('~-+~', '-', $text);
        $text = strtolower($text);
        return $text ?: 'n-a';
    }

    foreach ($listings as $l) {
        $uuid = bin2hex(random_bytes(16)); // PHP UUID or use SQL
        $uuid = substr($uuid, 0, 8) . '-' . substr($uuid, 8, 4) . '-' . substr($uuid, 12, 4) . '-' . substr($uuid, 16, 4) . '-' . substr($uuid, 20);
        $slug = slugify($l['title']) . '-' . substr($uuid, 0, 4);
        
        // Map status
        $status = 'active';
        if ($l['status'] == 'pending') $status = 'pending_approval';
        else if ($l['status'] == 'rejected') $status = 'suspended';

        $insertStmt->execute([
            $uuid,
            $l['company_uuid'],
            $l['pt_uuid'],
            $l['title'],
            $slug,
            $l['description'],
            $l['price'],
            $status,
            $l['type'] == 'sell' ? 'sale' : 'rent',
            $l['user_uuid'] ?: 'unknown-fix-later', // We'll need real user UUIDs
            $l['created_at']
        ]);
        
        // Migrate multi-brands
        $brandsStmt = $db->prepare("SELECT b.uuid FROM listing_brands lb JOIN brands b ON lb.brand_id = b.id WHERE lb.listing_id = ?");
        $brandsStmt->execute([$l['id']]);
        $brands = $brandsStmt->fetchAll();
        foreach ($brands as $b) {
            $db->prepare("INSERT INTO product_brands (product_id, brand_id) VALUES (?, ?)")->execute([$uuid, $b['uuid']]);
        }
    }

    $db->commit();
    echo "Data migration to products table completed.\n";
    
    // Safety check for created_by
    $db->exec("UPDATE products SET created_by = (SELECT uuid FROM users WHERE role = 'super_admin' LIMIT 1) WHERE created_by = 'unknown-fix-later'");

} catch (Exception $e) {
    if ($db->inTransaction()) $db->rollBack();
    echo "Error: " . $e->getMessage();
}
