<?php
namespace Models;

class Listing {
    public static function getByCompany($companyUuid) {
        $db = \Core\Database::getInstance();
        $stmt = $db->prepare("
            SELECT p.*, pt.name as category_name, c.name as company_name,
                   GROUP_CONCAT(b.name SEPARATOR ', ') as brand_names
            FROM products p
            JOIN companies c ON p.company_id = c.uuid
            LEFT JOIN product_types pt ON p.product_type_id = pt.uuid
            LEFT JOIN product_brands pb ON p.id = pb.product_id
            LEFT JOIN brands b ON pb.brand_id = b.uuid
            WHERE p.company_id = ? AND p.deleted_at IS NULL
            GROUP BY p.id
            ORDER BY p.created_at DESC
        ");
        $stmt->execute([$companyUuid]);
        return $stmt->fetchAll();
    }
    
    public static function getAll($companyUuid = null) {
        if ($companyUuid) return self::getByCompany($companyUuid);

        $db = \Core\Database::getInstance();
        $stmt = $db->query("
            SELECT p.*, pt.name as category_name, c.name as company_name,
                   GROUP_CONCAT(b.name SEPARATOR ', ') as brand_names
            FROM products p
            JOIN companies c ON p.company_id = c.uuid
            LEFT JOIN product_types pt ON p.product_type_id = pt.uuid
            LEFT JOIN product_brands pb ON p.id = pb.product_id
            LEFT JOIN brands b ON pb.brand_id = b.uuid
            WHERE p.deleted_at IS NULL
            GROUP BY p.id
            ORDER BY p.created_at DESC
        ");
        return $stmt->fetchAll();
    }

    public static function getPaginated($filters = [], $limit = 10, $offset = 0) {
        $db = \Core\Database::getInstance();
        
        $sql = "
            SELECT p.*, pt.name as category_name, c.name as company_name,
                   GROUP_CONCAT(b.name SEPARATOR ', ') as brand_names,
                   (SELECT image_path FROM product_images WHERE product_id = p.id AND is_primary = 1 LIMIT 1) as primary_image,
                   (SELECT image_path FROM product_images WHERE product_id = p.id ORDER BY is_primary DESC, sort_order ASC LIMIT 1) as fallback_image
            FROM products p
            JOIN companies c ON p.company_id = c.uuid
            LEFT JOIN product_types pt ON p.product_type_id = pt.uuid
            LEFT JOIN product_brands pb ON p.id = pb.product_id
            LEFT JOIN brands b ON pb.brand_id = b.uuid
            WHERE p.deleted_at IS NULL
        ";

        $params = [];
        if (!empty($filters['company_id'])) {
            $sql .= " AND p.company_id = ?";
            $params[] = $filters['company_id'];
        }
        
        if (!empty($filters['type'])) {
            if ($filters['type'] === 'buy') {
                $sql .= " AND p.listing_type IN ('sale', 'both')";
            } elseif ($filters['type'] === 'rent') {
                $sql .= " AND p.listing_type IN ('rent', 'both')";
            }
        }

        $sql .= " GROUP BY p.id ORDER BY p.created_at DESC LIMIT " . intval($limit) . " OFFSET " . intval($offset);
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public static function getTotalCount($filters = []) {
        $db = \Core\Database::getInstance();
        $sql = "SELECT COUNT(*) FROM products WHERE deleted_at IS NULL";
        $params = [];

        if (!empty($filters['company_id'])) {
            $sql .= " AND company_id = ?";
            $params[] = $filters['company_id'];
        }

        if (!empty($filters['type'])) {
            if ($filters['type'] === 'buy') {
                $sql .= " AND listing_type IN ('sale', 'both')";
            } elseif ($filters['type'] === 'rent') {
                $sql .= " AND listing_type IN ('rent', 'both')";
            }
        }

        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchColumn();
    }

    public static function getStats($companyId) {
        $db = \Core\Database::getInstance();
        $stmt = $db->prepare("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN listing_type IN ('sale', 'both') THEN 1 ELSE 0 END) as sell_count,
                SUM(CASE WHEN listing_type IN ('rent', 'both') THEN 1 ELSE 0 END) as rent_count
            FROM products 
            WHERE company_id = (SELECT uuid FROM companies WHERE id = ? OR uuid = ?)
            AND deleted_at IS NULL
        ");
        $stmt->execute([$companyId, $companyId]);
        return $stmt->fetch();
    }

    public static function count($status = null) {
        $db = \Core\Database::getInstance();
        $sql = "SELECT COUNT(*) FROM products WHERE deleted_at IS NULL";
        if ($status) {
            $stmt = $db->prepare($sql . " AND status = ?");
            $stmt->execute([$status]);
        } else {
            $stmt = $db->query($sql);
        }
        return $stmt->fetchColumn();
    }

    public static function find($id) {
        $db = \Core\Database::getInstance();
        $stmt = $db->prepare("
            SELECT p.*, pt.name as category_name, c.name as company_name,
                   GROUP_CONCAT(b.name SEPARATOR ', ') as brand_names
            FROM products p
            JOIN companies c ON p.company_id = c.uuid
            LEFT JOIN product_types pt ON p.product_type_id = pt.uuid
            LEFT JOIN product_brands pb ON p.id = pb.product_id
            LEFT JOIN brands b ON pb.brand_id = b.uuid
            WHERE p.id = ?
            GROUP BY p.id
        ");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    public static function getBrands($id) {
        $db = \Core\Database::getInstance();
        $stmt = $db->prepare("
            SELECT b.* 
            FROM brands b
            JOIN product_brands pb ON b.uuid = pb.brand_id
            WHERE pb.product_id = ?
        ");
        $stmt->execute([$id]);
        return $stmt->fetchAll();
    }

    public static function create($data) {
        $db = \Core\Database::getInstance();
        $db->beginTransaction();
        try {
            $id = self::generateUuid();
            $slug = self::slugify($data['title']) . '-' . substr($id, 0, 8);
            
            $sql = "INSERT INTO products (
                id, company_id, product_type_id, model_name, title, slug, description,
                `condition`, manufacture_year, country_of_origin,
                listing_type, status,
                warranty_applicable, warranty_type, warranty_duration_value, warranty_duration_unit,
                sale_stock_qty, rental_unit_qty,
                price, discount_type, discount_value,
                specification_doc, video_link,
                created_by, created_at
            ) VALUES (
                ?, ?, ?, ?, ?, ?, ?,
                ?, ?, ?,
                ?, ?,
                ?, ?, ?, ?,
                ?, ?,
                ?, ?, ?,
                ?, ?,
                ?, NOW()
            )";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $id,
                $data['company_id'], // must be uuid
                $data['product_type_id'], // required - category
                $data['model_name'] ?? null,
                $data['title'],
                $slug,
                $data['description'] ?? null,
                $data['condition'] ?? 'new',
                $data['manufacture_year'] ?: null,
                $data['country_of_origin'] ?? null,
                $data['listing_type'] ?? 'sale',
                $data['status'] ?? 'draft',
                $data['warranty_applicable'] ?? 0,
                $data['warranty_type'] ?? 'none',
                $data['warranty_duration_value'] ?: null,
                $data['warranty_duration_unit'] ?? null,
                $data['sale_stock_qty'] ?? 0,
                $data['rental_unit_qty'] ?? 0,
                $data['price'] ?: null,
                $data['discount_type'] ?? 'none',
                $data['discount_value'] ?: null,
                $data['specification_doc'] ?? null,
                $data['video_link'] ?? null,
                $data['created_by'] // must be uuid
            ]);

            if (!empty($data['brands'])) {
                $stmt = $db->prepare("INSERT INTO product_brands (product_id, brand_id) VALUES (?, ?)");
                foreach ($data['brands'] as $brandId) {
                    $stmt->execute([$id, $brandId]);
                }
            }

            $db->commit();
            return $id;
        } catch (\Exception $e) {
            $db->rollBack();
            error_log($e->getMessage());
            return false;
        }
    }

    public static function update($id, $data) {
        $db = \Core\Database::getInstance();
        $db->beginTransaction();
        try {
            $sql = "UPDATE products SET 
                product_type_id = ?, model_name = ?, title = ?, description = ?, 
                `condition` = ?, manufacture_year = ?, country_of_origin = ?,
                listing_type = ?, status = ?,
                warranty_applicable = ?, warranty_type = ?, 
                warranty_duration_value = ?, warranty_duration_unit = ?,
                sale_stock_qty = ?, rental_unit_qty = ?,
                price = ?, discount_type = ?, discount_value = ?,
                specification_doc = ?, video_link = ?,
                updated_by = ?, updated_at = NOW()
                WHERE id = ?";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $data['product_type_id'], // required - category
                $data['model_name'] ?? null,
                $data['title'],
                $data['description'] ?? null,
                $data['condition'] ?? 'new',
                $data['manufacture_year'] ?: null,
                $data['country_of_origin'] ?? null,
                $data['listing_type'] ?? 'sale',
                $data['status'] ?? 'draft',
                $data['warranty_applicable'] ?? 0,
                $data['warranty_type'] ?? 'none',
                $data['warranty_duration_value'] ?: null,
                $data['warranty_duration_unit'] ?? null,
                $data['sale_stock_qty'] ?? 0,
                $data['rental_unit_qty'] ?? 0,
                $data['price'] ?: null,
                $data['discount_type'] ?? 'none',
                $data['discount_value'] ?: null,
                $data['specification_doc'] ?? null,
                $data['video_link'] ?? null,
                $data['updated_by'],
                $id
            ]);

            // Sync multi-brands
            $db->prepare("DELETE FROM product_brands WHERE product_id = ?")->execute([$id]);
            if (!empty($data['brands'])) {
                $stmt = $db->prepare("INSERT INTO product_brands (product_id, brand_id) VALUES (?, ?)");
                foreach ($data['brands'] as $brandId) {
                    $stmt->execute([$id, $brandId]);
                }
            }

            $db->commit();
            return true;
        } catch (\Exception $e) {
            $db->rollBack();
            error_log($e->getMessage());
            return false;
        }
    }

    public static function delete($id) {
        $db = \Core\Database::getInstance();
        $stmt = $db->prepare("UPDATE products SET deleted_at = NOW() WHERE id = ?");
        return $stmt->execute([$id]);
    }

    private static function generateUuid() {
        $data = random_bytes(16);
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    private static function slugify($text) {
        $text = preg_replace('~[^\pL\d]+~u', '-', $text);
        $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
        $text = preg_replace('~[^-\w]+~', '', $text);
        $text = trim($text, '-');
        $text = preg_replace('~-+~', '-', $text);
        $text = strtolower($text);
        return $text ?: 'n-a';
    }
}
